<?php
class AuthController {
    private $userModel;
    private $faucetpayApiKey;

    public function __construct() {
        $this->userModel = new UserModel();
        $config = new ConfigModel();
        $this->faucetpayApiKey = $config->get('faucetpay_api_key_1') ?: '';
    }
    public function login() {
        if ($_POST) {
            // --- Rate limiting: configurable via settings ---
            $config = new ConfigModel();
            $maxAttempts = (int)($config->get('login_rate_limit_max_attempts') ?: 5);
            $windowSeconds = (int)($config->get('login_rate_limit_window_seconds') ?: 300);
            $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
            $now = time();
            if (!isset($_SESSION['login_rate_limit'])) {
                $_SESSION['login_rate_limit'] = [];
            }
            // Clean up old attempts
            $_SESSION['login_rate_limit'] = array_filter(
                $_SESSION['login_rate_limit'],
                function($entry) use ($now, $windowSeconds) { return $entry['ts'] > $now - $windowSeconds; }
            );
            // Count attempts from this IP
            $attempts = 0;
            foreach ($_SESSION['login_rate_limit'] as $entry) {
                if ($entry['ip'] === $ip) $attempts++;
            }
            if ($attempts >= $maxAttempts) {
                $_SESSION['error'] = 'Too many login attempts. Please wait ' . floor($windowSeconds/60) . ' minutes and try again.';
                header('Location: /');
                exit;
            }
            $_SESSION['login_rate_limit'][] = ['ip' => $ip, 'ts' => $now];
            // CSRF védelem
            if (
                empty($_SESSION['csrf_token']) ||
                empty($_POST['csrf_token']) ||
                !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])
            ) {
                $_SESSION['error'] = 'Invalid session token. Please reload the page.';
                header('Location: /');
                exit;
            }
            // Bot protection: honeypot field
            if (!empty($_POST['website'])) {
                $_SESSION['error'] = 'Bot detected!';
                header('Location: /');
                exit;
            }
            $email = $_POST['email'] ?? '';
            $referrerId = null;
            if (isset($_GET['ref'])) {
                $ref = preg_replace('/[^a-zA-Z0-9]/', '', $_GET['ref']);
                $refUser = $this->userModel->getUserByReferralCode($ref);
                if ($refUser) $referrerId = $refUser['id'];
            }
            $email = trim($email);
            $parts = explode('@', $email);
            if (count($parts) !== 2 || strtolower($parts[1]) !== 'gmail.com') {
                $_SESSION['error'] = 'Only Gmail addresses are allowed!';
                header('Location: /');
                exit;
            }
            $local = $parts[0];
            // Allow 0 or 1 dot, but not more than 1
            if (substr_count($local, '.') > 1) {
                $_SESSION['error'] = 'Your Gmail address can contain at most one dot before @gmail.com!';
                header('Location: /');
                exit;
            }
            $user = $this->userModel->getUserByEmail($email);
            if ($user) {
                // BANNED USER CHECK
                if (!empty($user['banned'])) {
                    $_SESSION['error'] = 'Your account has been banned. Please contact support.';
                    header('Location: /');
                    exit;
                }
                // LOGIN: user exists, just log in (do not check FaucetPay)
                session_regenerate_id(true); // Session fixation védelem
                $_SESSION['user_id'] = $user['id'];
                $_SESSION['user_email'] = $user['email'];
                $this->userModel->updateLastLogin($user['id']);
                header('Location: /dashboard');
                exit;
            }
            // REGISTER: user does not exist, check FaucetPay
            if (!$this->userModel->validateFaucetPayEmail($email, $this->faucetpayApiKey)) {
                $_SESSION['error'] = 'This Gmail address is not registered on FaucetPay!';
                header('Location: /');
                exit;
            }
            if ($this->userModel->createUser($email, $referrerId)) {
                $user = $this->userModel->getUserByEmail($email);
                if ($user) {
                    session_regenerate_id(true); // Session fixation védelem
                    $_SESSION['success'] = 'Registration successful!';
                    $_SESSION['user_id'] = $user['id'];
                    $_SESSION['user_email'] = $user['email'];
                    $this->userModel->updateLastLogin($user['id']);
                    header('Location: /dashboard');
                    exit;
                } else {
                    $_SESSION['error'] = 'Registration failed (user not found after insert)!';
                    header('Location: /');
                    exit;
                }
            } else {
                $_SESSION['error'] = 'Registration failed!';
                header('Location: /');
                exit;
            }
        }
        header('Location: /');
    }
    public function register() {
        header('Location: /');
    }
    public function logout() {
        // Csak a felhasználói session kulcsokat töröljük, az admin session-t NEM!
        unset($_SESSION['user_id'], $_SESSION['user_email'], $_SESSION['error'], $_SESSION['success']);
        // A CSRF token maradhat, vagy újragenerálható, de az admin session-t ne töröld!
        session_regenerate_id(true);
        header('Location: /');
    }
}