<?php
class DashboardController {
    public function index() {
        if (!isset($_SESSION['user_id'])) {
            header('Location: /');
            exit;
        }
        $userModel = new UserModel();
        $user = $userModel->getUserById($_SESSION['user_id']);

        // Ensure referral code exists
        if (empty($user['referral_code'])) {
            $referral_code = $userModel->generateReferralCode();
            // Save to DB
            $db = DatabaseConnect::getInstance()->getConnection();
            $stmt = $db->prepare("UPDATE users SET referral_code = ? WHERE id = ?");
            $stmt->execute([$referral_code, $user['id']]);
            $user['referral_code'] = $referral_code;
        }

        // Use only active faucet claims for GHZ (do NOT use users.ghz)
        $activeGhz = $userModel->getTotalActiveFaucetGhz($user['id']);
        $activeGhz = round($activeGhz, 4);

        // USDT mining rate: 1 GHZ = 0.000001 USDT/sec
        $config = new ConfigModel();
        $rate_per_ghz = $config->get('mining_rate_per_ghz') ?: '0.000001';
        $dashboardBanner = $config->get('dashboard_banner_html') ?: '';
        $siteTitle = $config->get('site_title') ?: 'Cloud Mining Platform';
        $apiKey1 = $config->get('faucetpay_api_key_1') ?: '';
        $apiKey2 = $config->get('faucetpay_api_key_2') ?: '';
        $currency = $config->get('selected_currency');
        $decimals = (int)($config->get('currency_decimals') ?: 8);

        // Calculate total uncollected USDT from all faucet claims
        $minedRaw = $userModel->getTotalUncollectedFaucetUsdt($user['id'], $rate_per_ghz, $decimals, true); // true: raw float
        $mined = number_format($minedRaw, $decimals, '.', '');

        // Collect
        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['collect'])) {
            // --- SECURITY: Only allow collect via POST, and only for the logged-in user ---
            // --- SECURITY: Do not trust client-side values, always recalculate on server! ---

            // CSRF protection (add this block)
            if (
                empty($_SESSION['csrf_token']) ||
                empty($_POST['csrf_token']) ||
                !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])
            ) {
                $_SESSION['error'] = "Invalid session token. Please reload the page.";
                header("Location: /dashboard");
                exit;
            }

            // Calculate collectable from server-side data only
            $minedRaw = $userModel->getTotalUncollectedFaucetUsdt($user['id'], $rate_per_ghz, $decimals, true);
            $minedRawStr = is_numeric($minedRaw) ? (string)$minedRaw : '0';
            if (stripos($minedRawStr, 'e') !== false) {
                $minedRawStr = sprintf('%.12f', $minedRaw);
            }
            $collectableStr = bcmul($minedRawStr, '100000000', 0);
            $collectableStr = bcdiv($collectableStr, '100000000', 8);

            error_log("COLLECT DEBUG - SECURE collectableStr (bcmath): $collectableStr for user_id: {$user['id']}");

            if ($collectableStr !== '0.00000000') {
                // --- SECURITY: Only update balance with server-calculated value ---
                $oldBalance = number_format((float)$user['balance'], 8, '.', '');
                $newBalance = bcadd($oldBalance, $collectableStr, 8);
                $userModel->updateUserBalanceAndCollection($user['id'], $newBalance, date('Y-m-d H:i:s'));
                $userModel->markFaucetClaimsCollected($user['id'], $rate_per_ghz, $decimals, (float)$collectableStr);
                $user = $userModel->getUserById($user['id']);
                // Regenerate CSRF token after successful collect
                $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
                $_SESSION['success'] = "You have successfully collected " . $collectableStr . " $currency!";
                header("Location: /dashboard");
                exit;
            } else {
                error_log("COLLECT DEBUG - REJECTED: collectableStr is 0.00000000 for user_id: {$user['id']}");
                $_SESSION['error'] = "No $currency to collect!";
                header("Location: /dashboard");
                exit;
            }
        }

        $dashboard = [
            'balance' => $user['balance'],
            'ghz' => number_format($userModel->getTotalActiveFaucetGhz($user['id']), 4, '.', ''),
            'last_collected_at' => $user['last_collected_at'],
            'mined' => $mined,
            'elapsed' => null,
            'from_ts' => null,
            'mining_rate' => $rate_per_ghz
        ];

        // Payout (withdraw) kezelése
        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['withdraw'])) {
            $userId = $_SESSION['user_id'];
            $userEmail = $_SESSION['user_email'];
            $userModel = new UserModel();
            $usdt = (float)$user['balance'];
            $satoshi = (int)round($usdt * 100000000);

            // Check if user is banned
            if (!empty($user['banned'])) {
                $_SESSION['error'] = 'Your account has been banned. Withdrawals are not allowed.';
                header('Location: /dashboard');
                exit;
            }

            $apiKeys = [];
            if ($apiKey1) $apiKeys[] = $apiKey1;
            if ($apiKey2) $apiKeys[] = $apiKey2;
            $success = false;
            $lastError = '';

            foreach ($apiKeys as $keyIndex => $apiKey) {
                $data = [
                    'api_key' => $apiKey,
                    'amount' => $satoshi,
                    'to' => $userEmail,
                    'currency' => $currency // use selected currency
                ];
                $opts = [
                    'http' => [
                        'header' => "Content-type: application/x-www-form-urlencoded\r\n",
                        'method' => 'POST',
                        'content' => http_build_query($data),
                        'timeout' => 10
                    ]
                ];
                $context = stream_context_create($opts);
                $result = @file_get_contents('https://faucetpay.io/api/v1/send', false, $context);

                if ($result !== false) {
                    $json = json_decode($result, true);
                    $errorMsg = isset($json['message']) ? strtolower($json['message']) : '';
                    // Log the error message for debugging
                    error_log("FaucetPay API Key #" . ($keyIndex+1) . " response: " . $errorMsg);

                    if (isset($json['status']) && $json['status'] == 200) {
                        $status = 'success';
                        $payoutId = $json['payout_id'] ?? null;
                        $currentBalance = is_numeric($user['balance']) ? number_format((float)$user['balance'], 8, '.', '') : '0.00000000';
                        $usdtStr = number_format($usdt, 8, '.', '');
                        $userModel->updateUserBalanceAndCollection($userId, bcsub($currentBalance, $usdtStr, 8), date('Y-m-d H:i:s'));
                        $userModel->logPayout($userId, $usdt, $satoshi, $status, $payoutId);

                        // Referral earnings: meghívó fél kapja a kifizetés 10%-át
                        $refUser = $userModel->getUserById($userId);
                        if ($refUser && $refUser['referrer_id']) {
                            $referralAmount = round($usdt * 0.10, 8);
                            if ($referralAmount > 0) {
                                $userModel->upsertReferralEarning($refUser['referrer_id'], $userId, $referralAmount);
                                $userModel->creditReferralEarningToBalance($refUser['referrer_id'], $referralAmount);
                            }
                        }

                        $_SESSION['success'] = "Withdrawal successful! Transaction ID: " . htmlspecialchars($payoutId);
                        $success = true;
                        break;
                    } else {
                        $lastError = $json['message'] ?? 'Unknown error';
                        // Check for any "insufficient" or "sufficient funds" error (balance, funds, etc.)
                        if (
                            strpos($errorMsg, 'insufficient') !== false ||
                            strpos($errorMsg, 'sufficient funds') !== false
                        ) {
                            // Try next API key
                            continue;
                        } else {
                            // Other error types - don't try next API key
                            break;
                        }
                    }
                } else {
                    $lastError = "API connection error";
                    continue;
                }
            }

            if (!$success) {
                $_SESSION['error'] = "Withdrawal failed: " . $lastError;
            }
            header("Location: /dashboard");
            exit;
        }

        // Payout history lekérése
        $payoutHistory = $userModel->getPayoutHistory($user['id']);

        // Referral
        $protocol = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') ? "https" : "http";
        $host = $_SERVER['HTTP_HOST'];
        $referralUrl = $protocol . "://{$host}/?ref=" . $user['referral_code'];
        // Only last 10 referrals
        $referrals = $userModel->getLastReferrals($user['id'], 10);

        // GHZ purchase hash check (referral reward)
        if (isset($_GET['deposit_hash'])) {
            $hashRaw = $_GET['deposit_hash'];
            $hash = ltrim(urldecode(trim($hashRaw)), '=');

            // file-based debug log
            $log = __DIR__ . '/../debug.log';
            @file_put_contents($log, sprintf("[%s] DASHBOARD hash raw:[%s] cleaned:[%s] user:%d\n", date('c'), $hashRaw, $hash, $_SESSION['user_id']), FILE_APPEND);

            $purchase = $userModel->getAnyGhzPurchaseByHash($hash);
            if ($purchase) {
                @file_put_contents($log, sprintf("[%s] DASHBOARD purchase found hash:[%s] status:%s user_id:%d ghz:%s\n", date('c'), $purchase['hash'], $purchase['status'], $purchase['user_id'], $purchase['ghz_amount']), FILE_APPEND);
                if ($purchase['user_id'] == $_SESSION['user_id']) {
                    if ($purchase['status'] === 'completed' || $purchase['status'] === 'pending') {
                        $ok1 = $userModel->increaseGhzAndCapacities($purchase['user_id'], $purchase['ghz_amount']);
                        // Referral reward
                        $refUser = $userModel->getUserById($purchase['user_id']);
                        if ($refUser && $refUser['referrer_id']) {
                            // 10% referral reward
                            $reward = round($purchase['ghz_amount'] * 0.10, 8);
                            if ($reward > 0) {
                                $userModel->addReferralReward($refUser['referrer_id'], $reward);
                            }
                        }
                        $userModel->markGhzPurchaseRewarded($hash);
                        @file_put_contents($log, sprintf("[%s] DASHBOARD credited ok:%s hash:%s\n", date('c'), var_export($ok1, true), $hash), FILE_APPEND);
                        $_SESSION['success'] = "Miner updated successfully! +" . number_format($purchase['ghz_amount'], 1) . " GHZ";
                    } elseif ($purchase['status'] === 'rewarded') {
                        $_SESSION['success'] = "This deposit was already credited.";
                    } else {
                        $_SESSION['error'] = "Deposit failed or not completed yet. (status: {$purchase['status']})";
                    }
                } else {
                    $_SESSION['error'] = "Deposit hash does not belong to this user.";
                    @file_put_contents($log, sprintf("[%s] DASHBOARD hash user mismatch hash_user:%d session_user:%d\n", date('c'), $purchase['user_id'], $_SESSION['user_id']), FILE_APPEND);
                }
            } else {
                $_SESSION['error'] = "Deposit not found for this hash [$hash].";
                @file_put_contents($log, sprintf("[%s] DASHBOARD purchase not found hash:[%s]\n", date('c'), $hash), FILE_APPEND);
            }

            // Always redirect so the page reloads fresh values and avoids double-processing
            header("Location: /dashboard");
            exit;
        }

        // Dashboard adat lekérés
        $referralEarnings = $userModel->getTotalReferralEarnings($user['id']);

        // If user has no active miner, show uncollected expired faucet USDT
        $showUncollectedExpired = false;
        $uncollectedExpiredUsdt = '0.00000000';
        if ($dashboard['ghz'] == '0.0000') {
            $uncollectedExpiredUsdt = $userModel->getTotalUncollectedExpiredFaucetUsdt($user['id'], $rate_per_ghz);
            if ($uncollectedExpiredUsdt > 0) {
                $showUncollectedExpired = true;
            }
        }

        // CSRF token for forms
        if (empty($_SESSION['csrf_token'])) {
            $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        }

        renderUser('dashboard', [
            'dashboard' => $dashboard,
            'payoutHistory' => $payoutHistory,
            'referralUrl' => $referralUrl,
            'referrals' => $referrals,
            'referralEarnings' => $referralEarnings,
            'showUncollectedExpired' => $showUncollectedExpired,
            'uncollectedExpiredUsdt' => $uncollectedExpiredUsdt,
            'dashboardBanner' => $dashboardBanner,
            'siteTitle' => $siteTitle,
            'currency' => $currency,
            'decimals' => $decimals, // pass decimals to view
            'csrf_token' => $_SESSION['csrf_token']
        ]);
    }
}

