<?php
class UserModel {
    private $db;
    public function __construct() {
        $this->db = DatabaseConnect::getInstance()->getConnection();
    }
    private function getUserIp() {
        if (!empty($_SERVER['HTTP_CF_CONNECTING_IP'])) {
            return $_SERVER['HTTP_CF_CONNECTING_IP'];
        }
        if (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            // Ha több IP van, az első a valódi
            $ips = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR']);
            return trim($ips[0]);
        }
        return $_SERVER['REMOTE_ADDR'] ?? null;
    }
    public function createUser($email, $referrerId = null) {
        $ghz = number_format(0.1, 1, '.', '');
        $referral_code = $this->generateReferralCode();
        $ip = $this->getUserIp();
        $stmt = $this->db->prepare("INSERT INTO users (email, balance, ghz, mining_started_at, referral_code, referrer_id, last_login_at, reklam_sent, last_ip, banned) VALUES (?, 0, ?, NOW(), ?, ?, NOW(), 0, ?, 0)");
        return $stmt->execute([$email, $ghz, $referral_code, $referrerId, $ip]);
    }
    public function getUserByReferralCode($code) {
        $stmt = $this->db->prepare("SELECT * FROM users WHERE referral_code = ?");
        $stmt->execute([$code]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    public function generateReferralCode() {
        return substr(bin2hex(random_bytes(5)), 0, 10);
    }
    public function getReferralList($userId) {
        $stmt = $this->db->prepare("SELECT * FROM users WHERE referrer_id = ?");
        $stmt->execute([$userId]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    public function addReferralReward($referrerId, $ghzReward) {
        // Increase GHZ for referrer (removed capacities)
        $stmt = $this->db->prepare("UPDATE users SET ghz = ghz + ? WHERE id = ?");
        return $stmt->execute([$ghzReward, $referrerId]);
    }
    public function getUserById($id) {
        $stmt = $this->db->prepare("SELECT * FROM users WHERE id = ?");
        $stmt->execute([$id]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($user && isset($user['ghz'])) {
            $user['ghz'] = number_format((float)$user['ghz'], 1, '.', '');
        }
        return $user;
    }
    public function updateUserBalanceAndCollection($userId, $newBalance, $collectedAt) {
        // Make sure $newBalance is a string with dot as decimal separator and max 8 decimals
        $newBalance = number_format((float)$newBalance, 8, '.', '');
        $stmt = $this->db->prepare("UPDATE users SET balance = ?, last_collected_at = ? WHERE id = ?");
        $stmt->execute([$newBalance, $collectedAt, $userId]);
        // Ensure the update succeeded
        return $stmt->rowCount() > 0;
    }
    public function validateFaucetPayEmail($email, $apiKey) {
        // Only allow gmail.com
        if (!is_string($email) || !preg_match('/^[^@]+@gmail\.com$/', $email)) {
            return false;
        }
        // FaucetPay API check
        $url = 'https://faucetpay.io/api/v1/checkaddress';
        $data = [
            'api_key' => $apiKey,
            'address' => $email
        ];
        $options = [
            'http' => [
                'header'  => "Content-type: application/x-www-form-urlencoded\r\n",
                'method'  => 'POST',
                'content' => http_build_query($data),
                'timeout' => 5
            ]
        ];
        $context  = stream_context_create($options);
        $result = @file_get_contents($url, false, $context);
        if ($result === FALSE) {
            return false;
        }
        $json = json_decode($result, true);
        if (isset($json['status']) && $json['status'] == 200 && !empty($json['payout_user_hash'])) {
            return true;
        }
        return false;
    }
    public function increaseGhzAndCapacities($userId, $ghzToAdd) {
        $ghzToAdd = (float)$ghzToAdd;
        if ($ghzToAdd <= 0) return false;
        $stmt = $this->db->prepare("UPDATE users SET ghz = ghz + ? WHERE id = ?");
        return $stmt->execute([$ghzToAdd, $userId]);
    }
    public function getTotalUserCount() {
        $stmt = $this->db->query("SELECT COUNT(*) FROM users");
        $real = (int)$stmt->fetchColumn();
        return $real;
    }
    public function getTotalWithdrawn() {
        $stmt = $this->db->query("SELECT SUM(amount_usdt) FROM payout_history WHERE status = 'success'");
        $real = (float)$stmt->fetchColumn();
        return $real;
    }
    public function getUserByEmail($email) {
        $stmt = $this->db->prepare("SELECT * FROM users WHERE email = ?");
        $stmt->execute([$email]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    // Only faucet_claims.ghz is used for mining, users.ghz is NOT used anymore
    public function getTotalActiveFaucetGhz($userId) {
        $stmt = $this->db->prepare("SELECT SUM(ghz) FROM faucet_claims WHERE user_id = ? AND expires_at > NOW()");
        $stmt->execute([$userId]);
        $sum = $stmt->fetchColumn();
        return $sum ? (float)$sum : 0.0;
    }
    public function addFaucetClaim($userId, $ghz, $minutes) {
        $stmt = $this->db->prepare("INSERT INTO faucet_claims (user_id, ghz, claimed_at, expires_at) VALUES (?, ?, NOW(), DATE_ADD(NOW(), INTERVAL ? MINUTE))");
        return $stmt->execute([$userId, $ghz, $minutes]);
    }
    public function getPayoutHistory($userId, $limit = 10) {
        $stmt = $this->db->prepare("SELECT * FROM payout_history WHERE user_id = ? ORDER BY id DESC LIMIT ?");
        $stmt->bindValue(1, $userId, PDO::PARAM_INT);
        $stmt->bindValue(2, (int)$limit, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    public function getLastReferrals($userId, $limit = 10) {
        $stmt = $this->db->prepare("SELECT * FROM users WHERE referrer_id = ? ORDER BY created_at DESC LIMIT ?");
        $stmt->bindValue(1, $userId, PDO::PARAM_INT);
        $stmt->bindValue(2, (int)$limit, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    public function logPayout($userId, $amountUsdt, $amountSatoshi, $status, $payoutId = null) {
        $stmt = $this->db->prepare("INSERT INTO payout_history (user_id, amount_usdt, amount_satoshi, status, payout_id) VALUES (?, ?, ?, ?, ?)");
        $stmt->execute([$userId, $amountUsdt, $amountSatoshi, $status, $payoutId]);
    }
    public function getFaucetClaimsForUser($userId) {
        $stmt = $this->db->prepare("SELECT * FROM faucet_claims WHERE user_id = ?");
        $stmt->execute([$userId]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    public function getTotalUncollectedFaucetUsdt($userId, $ratePerGhz = 0.000001, $decimals = 8, $raw = false) {
        $claims = $this->getFaucetClaimsForUser($userId);
        $now = time();
        $total = 0.0;
        foreach ($claims as $claim) {
            $claimedAt = strtotime($claim['claimed_at']);
            $expiresAt = strtotime($claim['expires_at']);
            $endTs = min($now, $expiresAt);
            $seconds = max(0, $endTs - $claimedAt);
            $maxUsdt = $seconds * (float)$claim['ghz'] * $ratePerGhz;
            $uncollected = $maxUsdt - (float)$claim['collected_usdt'];
            if ($uncollected > 0) $total += $uncollected;
        }
        if ($raw) return $total;
        return number_format($total, $decimals, '.', '');
    }
    public function markFaucetClaimsCollected($userId, $ratePerGhz = 0.000001, $decimals = 8, $collectAmount = null) {
        $claims = $this->getFaucetClaimsForUser($userId);
        $now = time();
        $remaining = is_null($collectAmount) ? null : (float)$collectAmount;
        foreach ($claims as $claim) {
            $claimedAt = strtotime($claim['claimed_at']);
            $expiresAt = strtotime($claim['expires_at']);
            $endTs = min($now, $expiresAt);
            $seconds = max(0, $endTs - $claimedAt);
            $maxUsdt = $seconds * (float)$claim['ghz'] * $ratePerGhz;
            $uncollected = $maxUsdt - (float)$claim['collected_usdt'];
            if ($uncollected > 0) {
                // Only collect up to 8 decimals, leave the rest for next time
                $collectable = $uncollected;
                if (!is_null($remaining)) {
                    // Truncate to 8 decimals
                    $collectable = min($collectable, $remaining);
                    $remaining -= $collectable;
                }
                $newCollected = (float)$claim['collected_usdt'] + $collectable;
                $stmt = $this->db->prepare("UPDATE faucet_claims SET collected_usdt = ? WHERE id = ?");
                $stmt->execute([number_format($newCollected, $decimals, '.', ''), $claim['id']]);
                if (!is_null($remaining) && $remaining <= 0) break;
            }
        }
    }

    public function getTotalReferralEarnings($userId) {
        $stmt = $this->db->prepare("SELECT SUM(amount) FROM referral_earnings WHERE user_id = ?");
        $stmt->execute([$userId]);
        $sum = $stmt->fetchColumn();
        return $sum ? number_format($sum, 8, '.', '') : '0.00000000';
    }
    public function creditReferralEarningToBalance($referrerId, $amount) {
        $amount = number_format($amount, 8, '.', '');
        $stmt = $this->db->prepare("UPDATE users SET balance = balance + ? WHERE id = ?");
        return $stmt->execute([$amount, $referrerId]);
    }
    public function getTotalUncollectedExpiredFaucetUsdt($userId, $ratePerGhz = 0.000001) {
        $claims = $this->getFaucetClaimsForUser($userId);
        $now = time();
        $total = 0.0;
        foreach ($claims as $claim) {
            $claimedAt = strtotime($claim['claimed_at']);
            $expiresAt = strtotime($claim['expires_at']);
            // Only expired claims
            if ($expiresAt <= $now) {
                $seconds = max(0, $expiresAt - $claimedAt);
                $maxUsdt = $seconds * (float)$claim['ghz'] * $ratePerGhz;
                $uncollected = $maxUsdt - (float)$claim['collected_usdt'];
                if ($uncollected > 0) $total += $uncollected;
            }
        }
        return number_format($total, 8, '.', '');
    }
    public function upsertReferralEarning($referrerId, $fromUserId, $amount) {
        $amount = number_format($amount, 8, '.', '');
        // Try to update existing row
        $stmt = $this->db->prepare("UPDATE referral_earnings SET amount = amount + ? WHERE user_id = ? AND from_user_id = ?");
        $stmt->execute([$amount, $referrerId, $fromUserId]);
        if ($stmt->rowCount() === 0) {
            // If no row was updated, insert new
            $stmt = $this->db->prepare("INSERT INTO referral_earnings (user_id, from_user_id, amount) VALUES (?, ?, ?)");
            $stmt->execute([$referrerId, $fromUserId, $amount]);
        }
    }
    public function updateLastLogin($userId) {
        $ip = $this->getUserIp();
        $stmt = $this->db->prepare("UPDATE users SET last_login_at = NOW(), last_ip = ? WHERE id = ?");
        return $stmt->execute([$ip, $userId]);
    }
}